#! /bin/bash

# (best viewed with tabs every 4 characters)

#   Copyright (C) 2009 Charles
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, US

# Filename: MyBackup.sh
	prg_ver='0.3'

# Purpose: 
#	Runs configurable backups using dar and, if possible, LVM snapshots then
#   writes files to DVD with par2 files for lost and damaged data recovery.

# Usage: 
#	See usage function below (search for "function usage") or use -h option.
#	See comments in sample configuration files distributed with this script.
	
# Environment:
#	Developed and tested on ubuntu 8.04.2 desktop 32-bit with
#	* bash 3.2-0ubuntu18
#	* dar and dar-static 2.3.5-1
# 	* dvd+rw-tools 7.0-9ubuntu1
#	* gnome-volume-manager 2.22.1-1ubuntu6
#	* lvm2 2.02.26-1ubuntu9 
#	* par2 0.4-9
#   Intended to work without par2 and probably does.
#   Intended to work without LVM but a few bugs likely.
#   May work with earlier and later versions of ubuntu and listed packages.

# Support: please contact Charles via 
# https://lists.sourceforge.net/lists/options/dar-support.

# History:
#	28jun8	Charles
#			* Begun!
#	2aug8	Charles
#			* Added ability to back up directory trees on non-LVM volumes.
#	9feb9	Charles
#			* Moved snapshot unmount and lvremove into function and call as soon 
#			  as snapshot no longer needed.
#			* Introduced cumulative error message in finalise function so it 
#			  does not give up on first error.
#	26mar9	Charles
#			* Introduced use of script command to both allow output from several 
#			  commands to go to screen as usual and capture it for analysis and 
#			  logging.
#			* Modified call_msg() and added -n option to msg() to allow 
#			  suppressing on-screen message.
#			* Added command line option to override config file's DVD device.
#	5apr9	Charles
#			Version 0.2
#			* Cleaned 
 characters from command screen output captured and 
#			  written to log.
#			* Added copyright and licence.
#			* Added removal of temporary DVD mount directory.
#			* Added log file name to final screen message.
#			* Changed dar --create screen output analysis to cope with more data 
#			  than bash can handle.
#			* Added options to msg() function to suppress writing to log, to 
#			  screen and timestamp in log.
#			* Added progress indicator when awk analysing dar --create output.
#	3jun9	Charles
#			Version 0.3
#			* Added par2 data redundancy feature.
# 			* Added command line option -V to show version and exit.
# 			* Added command line option -D for DVD control (only formatting now).
#			* Added configuration option 'Message'.
#			* Removed configuration option 'DVD mount directory'.
#			* Removed configuration option 'Snapshot mount base directory'.
#			* Introduced defaults for backup directory and DVD device so no 
#			  explicit global settings now required in config file.
#			* Added silent backup directory creation if it does not exist (to 
#			  facilitate setup).
#			* Added support comment, referencing dar mailing list.
#			* Added warning message category, 'W' and, if there are any warnings, 
#			  final error message to say say so and ensure return value at least 1.
#			* Changed category of unexpected output from growisofs from error to 
#			  warning.
#			* Changed to single temporary directory for all temporary files and 
#			  directories.
#			* Added /etc/lvm existence test before writing it to DVD.
#			* Added more error trappping in parse_cfg() function.
#			* Added DVD device file check.
#			* Added setting umask to 0077.
#			* Restructured DVD functions.
#			* Added, after error while mounting or reading DVD, copying last 300 
#			  lines of kern.log to screen and log.
#			* Removed (incomplete and untested) provisions for non-interactive 
#		  	  running.
#			* Removed ck_dir() function; the functionality is provided by 
#			  ck_file() function with :d
#			* Added block-special checking to ck_file() function.
#			* Added tty_msg() function.

# Wishlist (in approx descending order of importance/triviality):
#	* User guide inc. installation and configuration, to be part of a 
#	  distribution archive with script, sample config and bash completion. 
#	* Review configuration file data fields with embedded spaces in space 
#	  separated lists and subsequent \ escaping.
# 	* Allow spaces in prune and exclude lists and in name of "Directory 
#	  tree (fs_root)".
#	* Add dar_cp? Yes -- as long as it links to common libraries.
#	* Add differential backups.
#	* Use awk to build prune list (too slow in bash when list is long) or show 
#	  progress indicator.
#	* Add log file age-out.
#	* When waiting for DVD load, sit in a delayed loop looking for it so user 
#	  does not have to press Enter.
#	* Fix Ctrl+C not effective when running external programs such as dar. How?!
#	* Do not format if not DVD[+-]RW.  This info can be got in ck_DVD_loaded().
#	  Not urgent because the format command fails gracefully.
#	* Allow more than one "Direct directory tree"
#	* If "Direct directory tree"s (inc /etc/lvm if applicable) are on LVM then 
#	  use LVM snapshot when writing them to DVD.
#	* Handle screen output from growisofs nicely -- show the user a single
#	  countdown line.
#	* Adapt to suit user without root privileges?
#	* Move history and programmers' notes into documentation (when out of Beta).
#	* Allow for no writing to DVD (suit differentials when they come).
#	* Change maximum line length to 80 or maybe a tad more.
#	* Allow long options (change from getopts to getopt).

# Programmers' notes: error and trap handling: 
#   * All errors are fatal and finalise() is called.
#   * At any time, a trapped event may transfer control to finalise().
#   * To allow finalise() to tidy up before exiting, changes that need to be undone are
#     noted with global variables named <change name>_flag and the data required
#     to undo those changes is kept in global variables.
#	* finalise() uses the same functions to undo the changes as are used when they are
#	  undone routinely.
#	* $finalising_flag is used to prevent recursive calls when errors are encountered while finalise() is running,

# Programmers' notes: logical volume names
#   * Block device names are /dev/mapper/<volume group>-<logical volume>. 
#   * LVM commands require /dev/<volume group>/<logical volume> names and the OS provides thee as symlinks to the block device names.
#   * For consistency this script uses the names required by LVM commands except when an actual block device is essential.

# Programmers' notes: variable names and values
#	* Directory name variables are called *_dir and have values ending in /
#	* File name variables are called *_afn have values beginning with /
#	* Logic flag variables are called *_flag and have values YES or NO
#	* $buf is a localised scratch buffer.
#	* $lf is a line feed.
#	* $squote is a single quote.

# Programmers' notes: maximum line length ruler
# -------+---------+---------+---------+---------+---------+---------+---------+
#        10        20        30        40        50        60        70        80

# Programmers' notes: function call tree
#	+
#	|
#	+-- initialise
#	|   |
#	|   +-- usage
#	|   |
#	|   +-- parse_cfg
#	|
#	+-- back_up_dirtrees
#	|   |
#	|   +-- get_dev_afn
#	|   |
#	|   +-- set_up_snapshot
#	|   |
#	|   +-- do_dar_create
#	|   |
#	|   +-- do_dar_diff
#	|   |
#	|   +-- do_par2_create
#	|
#	+-- ensure_DVD_loaded
#	|   |
#	|   +-- ck_DVD_loaded
#	|
#	+-- umount_DVD
#	|
#	+-- format_DVD
#	|
#	+-- write_DVD
#	|
#	+-- mount_DVD
#	|
#	+-- verify_DVD
#	|
#	+-- finalise
#
# Utility functions called from various places:
# 	call_msg ck_file drop_snapshot msg report_dar_retval

# Function definitions in alphabetical order.  Execution begins after the last function definition.

#--------------------------
# Name: back_up_dirtrees
# Purpose: Does the dar jobs
# Usage: back_up_dirtrees 
#--------------------------
function back_up_dirtrees {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local buf idx

	# For each backup job defined in the config file ...
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	idx=1
	while [[ $idx -le "$n_dar_jobs" ]]
	do
		# Display any configured message
		# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		if [[ ${message[ $idx ]} != '' ]]; then
			tty_msg "${message[ $idx ]}"
		fi
		
		# Set up snapshot if on an LVM volume
		# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		get_dev_afn $idx
		if [[ ${dev_afn#/dev/mapper/} != $dev_afn ]]; then
	   		call_msg 'I' "Directory tree to be backed up, ${cfg_fs_roots[ $idx ]}, is on logical volume '$dev_afn'.  Using LVM snapshot"
			set_up_snapshot $idx
		else
			snapshot_mnt_points[ $idx ]=""
		fi

		# Remove any old par2 files
		# ~~~~~~~~~~~~~~~~~~~~~~~~~
		buf="$( $rm "$bk_dir${basenames[ $idx ]}.1.dar"*'.par2' 2>&1 )"
		case "$buf" in
			*'No such file or directory' | '' )
				;;
			* )
	   			call_msg 'W' "Problem removing old backup file(s). Output from $rm was '$buf'"
		esac

		# Finally -- the dar stuff!
		# ~~~~~~~~~~~~~~~~~~~~~~~~~
		do_dar_create $idx
		do_dar_diff $idx

		# Drop any LVM snapshot
		# ~~~~~~~~~~~~~~~~~~~~~
		if [[ ${snapshot_vol_mounted_flag[ $idx ]:-} = 'YES' ]]; then
			drop_snapshot $idx
		fi

		# par2 data redundancy files creation
		# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		if [[ ${par2_flag[ $idx ]:-} = 'YES' ]]; then
			do_par2_create $idx
		fi

		: $(( idx++ ))
	done

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of back_up_dirtrees

#--------------------------
# Name: call_msg
# Purpose: calls msg() and, for error messges, if finalise() has not been called, calls it
# $1 - message class
# $2 - message
#--------------------------
function call_msg {

   	msg -l $log_afn "$@"

	if [[ $1 = 'E' ]]; then
		press_enter_to_continue 				# Ensure user has chance to see error message
		if [[ ${finalising_flag:-} != 'YES' ]]; then
			finalise 1
		fi
	fi

	return 0

}  # end of function call_msg

#--------------------------
# Name: ck_DVD_loaded
# Purpose: Checks whether a recordable or rewriteable DVD is loaded
# Usage: ck_DVD_loaded 
# Return value:  0 if recordable or rewriteable DVD is loaded, 1 otherwise
#--------------------------
function ck_DVD_loaded {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local buf test

	buf="$( $dvd_rw_mediainfo "$DVD_dev" 2>&1 )"
	test="${buf##*Mounted Media:*( )}"			# strip up to mounted media data 
    test="${test%%$lf*}"						# strip remaining lines
	case "$test" in
		*'DVD'[+-]'R'* )
			call_msg 'I' "Media info:$lf$( echo "$buf" | $grep 'Media' )"
			fct "${FUNCNAME[ 0 ]}" 'returning 0'
			return 0	
			;;
	esac

	fct "${FUNCNAME[ 0 ]}" 'returning 1'
	return 1

}  # end of ck_DVD_loaded

#--------------------------
# Name: ck_file
# Purpose: for each file listed in the argument list: checks that it is 
#   reachable, exists and that the user has the listed permissions
# Usage: ck_file [ file_name:<filetype><permission>... ]
#	where 
#		file_name is a file name
#		type is b (block special file), f (file) or d (directoy)
#		permissions are one or more of r, w and x.  For example rx
# Returns: a bit screwy; tries to be clever but design intention unclear. Non-zero on error.
#--------------------------
function ck_file {

    local buf perm perms retval filetype

    # For each file ...
    # ~~~~~~~~~~~~~~~~~
    retval=0
    while [[ ${#} -gt 0 ]]
    do
	    file=${1%:*}
	    buf=${1#*:}
		filetype="${buf:0:1}"
	    perms="${buf:1:1}"
	    shift

	    if [[ $file = $perms ]]; then
		    echo "$prgnam: ck_file: no permisssions requested for file '$file'" >&2 
		    return 1
	    fi

	    # Is the file reachable and does it exist?
	    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		case $filetype in
			b )
	    		if [[ ! -b $file ]]; then
					echo "file '$file' is unreachable, does not exist or is not a block special file" >&2
					: $(( retval=retval+1 ))
					continue
	   		 	fi
				;;
			f )
	    		if [[ ! -f $file ]]; then
					echo "file '$file' is unreachable, does not exist or is not an ordinary file" >&2
					: $(( retval=retval+1 ))
					continue
	   		 	fi
				;;
			d )
	    		if [[ ! -d $file ]]; then
					echo "directory '$file' is unreachable, does not exist or is not a directory" >&2
					: $(( retval=retval+1 ))
					continue
	   		 	fi
				;;
			* )
		    	echo "$prgnam: ck_file: invalid filetype '$filetype' specified for file '$file'" >&2 
		    	return 1
		esac

	    # Does the file have the requested permissions?
	    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	    buf="$perms"
	    while [[ $buf != '' ]]
	    do
		    perm="${buf:0:1}"
		    buf="${buf:1}"
            case $perm in
	    		r )	
	    		    if [[ ! -r $file ]]; then
	    		        echo "file '$file' does not have requested read permission" >&2
	    		        let retval=retval+1
	    		        continue
	    		    fi
                    ;;
		    	w )
		    		if [[ ! -w $file ]]; then
		    			echo "file '$file' does not have requested write permission" >&2
		    			let retval=retval+1
		    			continue
		    		fi
		    		;;
		    	x )
		    		if [[ ! -x $file ]]; then
		    			echo "file '$file' does not have requested execute permission" >&2
		    			let retval=retval+1
		    			continue
		    		fi
		    		;;
		    	* )
		    		echo "$prgnam: ck_file: unexpected permisssion '$perm' requested for file '$file'" >&2
		    		return 1
		    esac
	    done

    done

return $retval

}  #  end of function ck_file

#--------------------------
# Name: do_dar_create
# Purpose: runs a single dar job
# Usage: do_dar_create idx
#	where 
#		idx is the index to a set of arrays of dar create job parameters
#--------------------------
function do_dar_create {

	fct "${FUNCNAME[ 0 ]}" "started with idx $1"

    local buf idx dar_cmd dir error_flag exclude_list log_msg log_msg_idx_max mask cfg_fs_root mnt prune_list retval dar_fs_root snapshot_mnt_point warning_flag

	idx="$1"

	# Build fs_root for use on dar command
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	cfg_fs_root="${cfg_fs_roots[ $idx ]}"				# fs_root as specified in the config file
	cfg_fs_root="${cfg_fs_root%%*(/)}/"					# ensure single trailing /
	if [[ ${snapshot_mnt_points[ $idx ]} = "" ]]; then
		# dirtree to back up is not on LV
		dar_fs_root="$cfg_fs_root"
	else
		# dirtree to back up is on LV
		buf="$( $df "$cfg_fs_root" 2>&1 )"				# No error trap because same command ran OK in get_dev_afn()
		mnt="${buf##* }"								# live file system mount point
		buf="${cfg_fs_root#$mnt/}"						# directory tree relative to mount point
		snapshot_mnt_point="${snapshot_mnt_points[ $idx ]}"
		dar_fs_root="${snapshot_mnt_point}$buf"
	fi
	# @@ Better to have a normalise path function to escape all special characters and to change // to /?
	dar_fs_root="${dar_fs_root// /\\ }"					# Escape any embedded spaces
	dar_fs_root="${dar_fs_root//\/\//\/}"				# Convert any // to /
	dar_fs_roots[ $idx ]="$dar_fs_root"

	# Build prune list
	# ~~~~~~~~~~~~~~~~
	buf="${prunes[ $idx ]##*( )}"
	prune_list=
	while [[ $buf != '' ]]
	do
		dir="${buf%%[ 	]*}"							# strip all after first space or tab
		buf="${buf#"$dir"}"								# remove directory just taken
		prune_list="$prune_list -P $dir "
		buf="${buf##*([ 	])}"						# strip leading spaces and tabs
	done

	# Build exclude list
	# ~~~~~~~~~~~~~~~~~~
	buf="${excludes[ $idx ]##*( )}"
	exclude_list=
	while [[ $buf != '' ]]
	do
		mask="${buf%%[ 	]*}"							# strip all after first space or tab
		buf="${buf#"$mask"}"							# remove mask just taken
		exclude_list="$exclude_list -X $mask "
		buf="${buf##*([ 	])}"						# strip leading spaces and tabs
	done

	# Build dar create command
	# ~~~~~~~~~~~~~~~~~~~~~~~~
	dar_cmd="$( echo \
	$dar --create "${bk_dir}${basenames[ $idx ]}" --fs-root $dar_fs_root \
    --alter=atime --empty-dir --mincompr 256 --noconf --no-warn --verbose \
    --prune lost+found \
    $prune_list \
    $exclude_list \
    --gzip=5 --alter=no-case \
	-Z '*.asf' \
	-Z '*.avi' \
	-Z '*.bzip' \
	-Z '*.bzip2' \
	-Z '*.divx' \
	-Z '*.gif' \
	-Z '*.gzip' \
	-Z '*.jpeg' \
	-Z '*.jpg' \
	-Z '*.mp3' \
	-Z '*.mpeg' \
	-Z '*.mpg' \
	-Z '*.png' \
	-Z '*.ra' \
	-Z '*.rar' \
	-Z '*.rm' \
	-Z '*.tgz' \
	-Z '*.wma' \
	-Z '*.wmv' \
	-Z '*.Z' \
	-Z '*.zip' \
	)"

	# Write a shellscript to run dar --create
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	buf="#! /bin/bash$lf$dar_cmd${lf}echo $dar_retval_preamble\$?$lf# End of script" > $tmp_sh_afn 	
	echo "$buf" > $tmp_sh_afn 	

	# Run the shellscript, capturing its screen output in a temporary file
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	call_msg 'I' "Running on-the-fly script to run dar backup.  Script is:$lf$buf"
	"$script" '-c' "$tmp_sh_afn" '-q' '-f' "$tmp_afn"

	# Analyse captured screen output
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# When the dar command produces a lot of output, it takes too long to process in bash so awk is used instead.
	# The awk program below analyses the dar screen output and writes bash variable assignment statements which are actioned by eval
	# @@ more efficient for awk to write dar --create output driectly to log, only passing unexpected lines back to bash?
	eval "$( $awk '
		function write_log_msg()
	     	{
				# This could be expanded to write log_bad_msg as well as log_msg
				log_msg_idx++
				log_msg = substr(log_msg, 2)						# Remove initial \n
				gsub(squote, squote dquote squote dquote squote, log_msg)
				gsub("\r", "", log_msg)
				print "log_msg[" log_msg_idx "]="squote log_msg squote
				log_msg = ""
			}

	    BEGIN {
				print "Analysing dar --create output for logging and error reporting" > "/dev/tty"
	            dquote = "\042"
				error_flag = "NO"
				log_msg_idx = -1
	            squote = "\047"
				warning_flag = "NO"
	    	}

		// \
			{
				# When there are too many lines in the log message this program is slow and has been seen to break (no log_msg output)
				# The solution is to write it 500 line chunks
				if ((NR % 500) > 498.1) 
				{
					printf "." > "/dev/tty"
					write_log_msg()
				}
			}

		/Return value from dar command is: / \
			{
				i = split($0, array)
				dar_retval = array[i]
				next
			}

		/^Removing file / \
		|| /^Adding file to archive: / \
		|| /^Writing archive contents\.\.\./ \
		|| /^\r$/ \
		|| /^Recording hard link to archive: / \
		|| / --------------------------------------------/ \
		|| /.* inode\(s\) saved/ \
		|| /.* with .* hard link\(s\) recorded/ \
		|| /.* inode\(s\) not saved \(no inode\/file change\)/ \
		|| /.* inode\(s\) ignored \(excluded by filters\)/ \
		|| /.* inode\(s\) recorded as deleted from reference backup/ \
		|| /^ Total number of inodes* considered: / \
		|| /^ EA saved for [0-9]* inode\(s\)/ \
			{														# Normal
				log_msg = log_msg "\n" $0
				next
			}
		
		/inode\(s\) changed at the moment of the backup/ \
			{														# Maybe normal, maybe unexpected
				split($0, array)
				if (array[1] == 0)				
				{
					log_msg = log_msg "\n" $0
				}
				else
				{
					log_msg = log_msg "\n" "'"$prgnam"' identified unexpected output: " $0
					log_bad_msg = log_bad_msg "\n" $0
					if ("'"${snapshot_mnt_points[ $idx ]}"'" == "") { error_flag = "YES" } else { warning_flag = "YES" }
				}
				next
			}

		/.* inode\(s\) failed to save \(filesystem error\)/ \
			{														# Maybe normal, maybe unexpected
				split($0, array)
				if (array[1] == 0)
				{													# Normal
					log_msg = log_msg "\n" $0
				}
				else
				{
					warning_flag = "YES" 							# Filesystem error but keep going with the backup		
					log_msg = log_msg "\n" "'"$prgnam"' identified unexpected output: " $0
					log_bad_msg = log_bad_msg "\n" $0
				}
				next
			}

		/.*/ \
			{
				warning_flag = "YES" 								# Unexpected output but keep going with the backup		
				log_msg = log_msg "\n" "'"$prgnam"' identified unexpected output: " $0
				log_bad_msg = log_bad_msg "\n" $0
				next
			}

		END {
				printf "\n" > "/dev/tty"
				log_msg = log_msg "\n" $0 							# Last line is " --------------------------------------------"
				write_log_msg()
				print "log_msg_idx_max=" log_msg_idx
				print "dar_retval=" dar_retval
				print "error_flag=" squote error_flag squote
				gsub(squote, squote dquote squote dquote squote, log_bad_msg)
				print "log_bad_msg=" squote log_bad_msg squote
				print "warning_flag=" squote warning_flag squote
			}' \
	$tmp_afn )"

	# Message and log as required
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# The following line only prints the first 500 lines. Too much data for bash?  Solution is to loop for each set of 500 lines
	# call_msg '-n' 'I' "dar --create output:${log_msg[@]}"
	call_msg -n s 'I' "dar --create output:"
	for (( idx = 0; idx <= log_msg_idx_max; idx++))
	do
		call_msg -n s -n t 'I' "${log_msg[$idx]}"
	done
	if [[ $error_flag = 'YES' ]]; then
		call_msg 'E' "Unexpected output from dar --create:$log_bad_msg"
	else 
		if [[ $warning_flag = 'YES' ]]; then
			call_msg 'W' "Unexpected output from dar --create:$log_bad_msg"
		fi
	fi
	report_dar_retval $dar_retval

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of do_dar_create

#--------------------------
# Name: do_dar_diff
# Purpose: checks dar backup file on disk against backed up files.
# Usage: do_dar_diff
# 		$1 - index number of dar job
#--------------------------
function do_dar_diff {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local buf dar_retval error_flag idx log_msg warning_flag

	idx="$1"

	# Write a shellscript
	# ~~~~~~~~~~~~~~~~~~~
	buf="#! /bin/bash$lf$dar --diff ${bk_dir}${basenames[ $idx ]} --fs-root ${dar_fs_roots[ $idx ]}${lf}echo $dar_retval_preamble\$?$lf# End of script" > $tmp_sh_afn 	
	echo "$buf" > $tmp_sh_afn 	

	# Run the shellscript, capturing its screen output in a temporary file
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	call_msg 'I' "Running on-the-fly script to verify dar backup on disk against backed up files.  Script is:$lf$buf"
	"$script" '-c' "$tmp_sh_afn" '-q' '-f' "$tmp_afn"

	# Analyse captured screen output
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	error_flag='NO'
	log_msg=''
	warning_flag='NO'
	exec 3< $tmp_afn 												# set up the temp file for reading on file descriptor 3
	while read -u 3 buf												# for each line of the script output to screen
	do
		buf="${buf%
}"
		case "$buf" in
			$dar_retval_preamble* )
				buf="${buf#$dar_retval_preamble}"
				dar_retval="$buf"
				;;
			'' \
			| *'--------------------------------------------'* \
			| *'inode(s) treated' \
			| *'inode(s) ignored (excluded by filters)' \
			| *'Total number of inode considered:'* ) 				# Normal output
				log_msg="$log_msg$lf$buf"
				;;
			*'inode(s) do not match those on filesystem' )			# Maybe normal, maybe unexpected
				log_msg="$log_msg$lf$buf"
				buf="${buf%% *}"
				if [[ "$buf" != '0' ]]; then
					if [[ "${snapshot_mnt_points[ $idx ]}" != '' ]]; then
						error_flag='YES'							# Unexpected with LVM snapshot
					else
						warning_flag='YES'							# Normal but iffy when not LVM snapshot
					fi
				fi
				;;
			* )														# Unexpected output
				error_flag='YES'
				log_msg="$log_msg${lf}$prgnam identified unexpected output: $buf"
				;;
		esac

	done
	exec 3<&- 														# free file descriptor 3

	# Message and log as required
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~
	if [[ $error_flag = 'NO' ]]; then 
		call_msg -n s 'I' "Verifed dar backup on disk against backed up files OK:$log_msg"
	else
		call_msg 'E' "Unexpected output when verifying dar backup on disk against backed up files:$log_msg"
	fi
	report_dar_retval $dar_retval
	if [[ $warning_flag == 'YES' ]]; then
		call_msg 'I' 'WARNING: some files had changed on disk (see above)'
	fi

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of do_dar_diff

#--------------------------
# Name: do_par2_create
# Purpose: runs par2 create job to create data redundancy files
# Usage: do_par2_create
# 		$1 - index number of dar job
#--------------------------
function do_par2_create {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local buf dar_retval error_flag idx log_msg warning_flag

	idx="$1"

	# Write a shellscript
	# ~~~~~~~~~~~~~~~~~~~
	# Decided not to use the -q option as it supresses assurance messages.
	buf="#! /bin/bash$lf$par2 create ${bk_dir}${basenames[ $idx ]}.1.dar$lf# End of script" > $tmp_sh_afn 	
	echo "$buf" > $tmp_sh_afn 	

	# Run the shellscript, capturing its screen output in a temporary file
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	call_msg 'I' "Running on-the-fly script to generate data redundancy files.  Script is:$lf$buf"
	"$script" '-c' "$tmp_sh_afn" '-q' '-f' "$tmp_afn"

	# Analyse captured screen output
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	error_flag='NO'
	log_msg=''
	warning_flag='NO'
	exec 3< $tmp_afn 												# set up the temp file for reading on file descriptor 3
	while read -u 3 buf												# for each line of the script output to screen
	do
		buf="${buf%
}"
		case "$buf" in
			'' \
			| 'par2cmdline version 0.4'* \
			| 'par2cmdline comes with'* \
			| 'This is free software'* \
			| 'it under the terms of the GNU'* \
			| 'Free Software Foundation'* \
			| 'option) any later version'* \
			| 'Block size:'* \
			| 'Source file count:'* \
			| 'Source block count:'* \
			| 'Redundancy:'* \
			| 'Recovery block count:'* \
			| 'Recovery file count:'* \
			| 'Opening:'* \
			| 'Computing Reed Solomon'* \
			| 'Wrote '* \
			| 'Writing '* \
			| 'Done' ) 												# Normal output
				log_msg="$log_msg$lf$buf"
				;;
			*'Processing: '* | 'Constructing:'* )					# Normal output, not wanted in log
				;; 
			* )														# Unexpected output
				error_flag='YES'
				log_msg="$log_msg${lf}++++$prgnam identified unexpected output: $buf"
				;;
		esac

	done
	exec 3<&- 														# free file descriptor 3

	# Message and log as required
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~
	if [[ $error_flag = 'NO' ]]; then 
		call_msg -n s 'I' "Generated data redundancy files OK:$log_msg"
	else
		call_msg 'W' "Unexpected output when generating data redundancy files (prefixed with ++++):$log_msg"
	fi

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of do_par2_create

#--------------------------
# Name: drop_snapshot
# Purpose: umnmounts and removes a snapshot
# Usage: drop_snapshot
# 		$1 - index number of dar job
# 		$2 - name of variable to put any error message in (only used when finalising)
#--------------------------
function drop_snapshot {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local buf idx my_msg

	idx="$1"

	# Unmount the snapshot
	# ~~~~~~~~~~~~~~~~~~~~
	buf="$( "$umount" "${snapshot_mnt_points[ $idx ]}" 2>&1 )"
	if [[ $? -eq 0 ]]; then
		snapshot_vol_mounted_flag[ $idx ]='NO'
		call_msg 'I' "Unmounted snapshot from ${snapshot_mnt_points[ $idx ]}."
	else
		my_msg="Unable to unmount snapshot from ${snapshot_mnt_points[ $idx ]}.  $umount output was$lf$buf"
		if [[ ${finalising_flag:-} = '' ]]; then
			call_msg 'E' "$my_msg"
		else
			eval "$2"="'$my_msg'"
			fct "${FUNCNAME[ 0 ]}" 'returning on umount error'
			return 1
		fi
	fi

	# Remove the snapshot
	# ~~~~~~~~~~~~~~~~~~~
	buf="$( $lvremove --force ${LVM_snapshot_devices[ $idx ]} 2>&1 )"
	if [[ $? -eq 0 ]]; then
		call_msg 'I' "Removed snapshot volume ${LVM_snapshot_devices[ $idx ]}."
	else
		my_msg="Unable to remove snapshot volume ${LVM_snapshot_devices[ $idx ]}.  $lvremove output was$lf$buf"
		if [[ ${finalising_flag:-} = '' ]]; then
			call_msg 'E' "$my_msg"
		else
			eval "$2"="'$my_msg'"
			fct "${FUNCNAME[ 0 ]}" 'returning on lvremove error'
			return 1
		fi
	fi

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of drop_snapshot

#--------------------------
# Name: ensure_DVD_loaded
# Purpose: Ensures a recordable or rewriteable DVD is loaded
# Usage: ensure_DVD_loaded 
# Return value:  0 if recordable or rewriteable DVD is loaded, 1 otherwise
#--------------------------
function ensure_DVD_loaded {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local buf test

	until ck_DVD_loaded
	do
		case "$restart" in
			'' | 'DVD_write' )
				msg="Please load a recordable or rewritable DVD in $DVD_dev."
				;;
			'DVD_verify' )
				msg="Please load the DVD already recorded for this backup job in $DVD_dev."
				;;
			* )
				call_msg 'E' "Programming error: unexpected restart value, '$restart'"
				;;
		esac
		tty_msg "$msg"
	done

	fct "${FUNCNAME[ 0 ]}" 'returning 1'
	return 1

}  # end of ensure_DVD_loaded

#--------------------------
# Name: fct
# Purpose: function call trace (for debugging)
# $1 - name of calling function 
# $2 - message.  If it starts with "started" or "returning" then the output is prettily indented
#--------------------------
function fct {

	if [[ $debug = 'NO' ]]; then
		return 0
	fi

	fct_ident="${fct_indent:=}"

	case $2 in
		'started'* )
			fct_indent="$fct_indent  "
			call_msg 'I' "DEBUG: $fct_indent$1: $2"
			;;
		'returning'* )
			call_msg 'I' "DEBUG: $fct_indent$1: $2"
			fct_indent="${fct_indent#  }"
			;;
		* )
			call_msg 'I' "DEBUG: $fct_indent$1: $2"
	esac

}  # end of function fct

#--------------------------
# Name: finalise
# Purpose: cleans up and gets out of here
#--------------------------
function finalise {
	fct "${FUNCNAME[ 0 ]}" 'started'

	local buf msg msgs my_retval retval

	finalising_flag='YES'
	
	# Set return value
	# ~~~~~~~~~~~~~~~~
	# Choose the highest and give message if finalising on a trapped signal
	my_retval="${prgnam_retval:-0}" 
	if [[ $1 -gt $my_retval ]]; then
		my_retval=$1
	fi
	case $my_retval in 
		129 | 130 | 131 | 143 )
			case $my_retval in
				129 )
					buf='SIGHUP'
					;;
				130 )
					buf='SIGINT'
					;;
				131 )
					buf='SIGQUIT'
					;;
				143 )
					buf='SIGTERM'
					;;
			esac
			call_msg 'I' "finalising on $buf"
			;;
	esac

	# Drop any snapshot(s)
	# ~~~~~~~~~~~~~~~~~~~~
	msgs=''
	i="$n_dar_jobs"
	while [[ $i -gt 0 ]]
	do
		if [[ ${snapshot_vol_mounted_flag[ $i ]:-} = 'YES' ]]; then
			drop_snapshot $i 'msg'
			retval=$?
			if [[ $retval -ne 0 ]]; then
				msgs="$msgs$lf$msg"
				if [[ $retval -gt $my_retval ]]; then
					my_retval=$retval
				fi
			fi
		fi
		: $(( i-- ))
	done

	# Unmount DVD
	# ~~~~~~~~~~~
	# At first eject was used to both unmount and eject but it was not reliable
	if [[ ${DVD_mounted_flag:-} = 'YES' ]]; then
		# Sometimes umount fails with "device is busy" so pause for a second before umounting
		$sleep 1
		buf="$( $umount $DVD_dev 2>&1 )"
		if [[ $? -eq 0 ]]; then
			buf="$( $eject $DVD_dev 2>&1 )"
			if [[ $? -eq 0 ]]; then
				call_msg 'I' 'DVD unmounted and ejected'
			else
				msgs="$msgs${lf}Unable to eject $DVD_dev.  $eject output was$lf$buf"
			fi
		else
			msgs="$msgs${lf}Unable to unmount $DVD_dev.  $umount output was$lf$buf"
		fi
	fi

	# Remove any temporary directory and contents
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	if [[ ${tmp_dir_created_flag:-} = 'YES' ]]; then
		$rm -fr $tmp_dir 
	fi

	# Final log messages
	# ~~~~~~~~~~~~~~~~~~
	if [[ $global_warning_flag = 'YES' ]]; then
		msgs="$msgs${lf}There were WARNINGs. Please search log for ' W ' (without the quotes) to ensure they are not significant for your backup"
		if [[ $my_retval -eq 0 ]]; then
			my_retval=1
		fi
	fi
	if [[ "$msgs" != '' ]]; then
		msgs="${msgs#$lf}"		# strip leading linefeed
		call_msg 'E' "$msgs"
	fi
	call_msg 'I' "Exiting with return value $my_retval"
	call_msg -n l 'I' "Log file: $log_afn"

	# Exit, ensuring exit is used, not any alias
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	fct "${FUNCNAME[ 0 ]}" 'exiting'
	\exit $my_retval

}  # end of function finalise

#--------------------------
# Name: format_DVD
# Purpose: formats the DVD
# Usage: format_DVD
#--------------------------
function format_DVD {

	fct "${FUNCNAME[ 0 ]}" "started"

    local buf error_flag log_msg

	# Formatting the DVD is precautionary.  Some DVDs, written on other systems, caused genisofs (called by 
	# growisofs) to error.  Experiments showed the problem could be worked around by formatting.

	# Write shellscript to format the DVD
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	buf="#! /bin/bash$lf$dvd_rw_format -force $DVD_dev$lf# End of script" > $tmp_sh_afn 	
	echo "$buf" > $tmp_sh_afn 	

	# Run the shellscript, capturing its screen output in a temporary file
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	call_msg 'I' "Running on-the-fly script to format DVD.  Script is:$lf$buf"
	"$script" '-c' "$tmp_sh_afn" '-q' '-f' "$tmp_afn"

	# Analyse captured screen output
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	error_flag='NO'
	log_msg=''
	exec 3< $tmp_afn 												# set up the temp file for reading on file descriptor 3
	while read -u 3 buf												# for each line of the script output to screen
	do
		buf="${buf%
}"
		case "$buf" in
			'* BD/DVD±RW/-RAM format utility'*  | '* 4.7GB DVD'* ) 	# Normal output
				log_msg="$log_msg$lf$buf"
				;;
			'* formatting'* )										# Progress reporting -- not required in log
				;;
			* )														# Unexpected output
				error_flag='YES'
				log_msg="$log_msg${lf}$prgnam identified unexpected output: $buf"
				;;
		esac

	done
	exec 3<&- 														# free file descriptor 3

	if [[ $error_flag = 'NO' ]]; then 
		call_msg -n s 'I' "Formatted DVD OK:$log_msg"
	else
		call_msg 'E' "Unexpected output when formatting DVD:$log_msg"
	fi

	call_msg 'I' "DEBUG for https://bugs.launchpad.net/ubuntu/+source/linux/+bug/228624: After formattng DVD, /sys/block/sr0/size contains: $(/bin/cat /sys/block/sr0/size)" 

	fct "${FUNCNAME[ 0 ]}" "returning"

}  # end of format_DVD

#--------------------------
# Name: get_dev_afn
# Purpose: gets the absolute filename of the device file of the filesystem the dirtree is
# Usage: get_dev_afn index
#	where 
#		index is the dar job index number
#--------------------------
function get_dev_afn {

	fct "${FUNCNAME[ 0 ]}" "started with idx=$1"

    local buf idx cfg_fs_root 

	# Get device name for file system containing the directory tree to back up
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	idx="$1"
	cfg_fs_root="${cfg_fs_roots[ $idx ]}"
	buf="$($df "$cfg_fs_root" 2>&1)"
	if [[ $? -ne 0 ]]; then
	    call_msg 'E' "Unable to get device name for file system with directory tree '$cfg_fs_root'.  $df output was$lf$buf" 
	fi
	# df wraps output if the device path is too long to fit in the "Filesystem" column
	buf="${buf#*$lf}"						# strip first line
	buf="${buf%$lf*}"						# strip possible third (now second) line
	dev_afn="${buf%% *}"					# strip possible space onward

	fct "${FUNCNAME[ 0 ]}" "returning. \$dev_afn is '$dev_afn'"

}  # end of get_dev_afn

#--------------------------
# Name: initialise
# Purpose: sets up environment and parses command line
#--------------------------
function initialise {

	fct "${FUNCNAME[ 0 ]}" 'started'

	local args cfg_fn emsg msgpart timestamp

	# Utility variables
	# ~~~~~~~~~~~~~~~~~
	dar_retval_preamble='Return value from dar command is: '
	lf=$'\n'							# ASCII linefeed, a.k.a newline
	squote="'"

	# Set defaults that may be overriden by command line parsing
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	cfg_dir="/etc/opt/$prgnam/" 		# can only be overridden on the command line
	DVD_dev='/dev/scd0'
	DVD_format_flag='NO'				# formatting is defensive but reduces DVD life
	emsg=''
	log_dir="/var/opt/$prgnam/log/"		# can only be overridden on the command line
	restart=''
	writer_opt_flag='NO'				# DVD writer option not given; use one specified in config file

	# Parse command line
	# ~~~~~~~~~~~~~~~~~~
	args="${*:-}"
	while getopts c:dD:hl:r:Vw: opt 2>/dev/null
	do
		case $opt in
			c)
				if [[ ${OPTARG:0:1} != '/' ]]; then
					cfg_afn="${cfg_dir}$OPTARG"
				else
					cfg_afn="$OPTARG"
				fi
				;;
			d )
				debug='YES'
				;;
			D )
				case "$OPTARG" in
					'format' ) 
						DVD_format_flag='YES'
						;;
					'no_format' ) 
						DVD_format_flag='NO'
						;;
					* )
						emsg="${lf}Option -D: invalid value '$OPTARG'"
						;;
				esac
				;;
			h )
				usage -v
				\exit 0
				;;
			l)
				if [[ ${OPTARG:0:1} != '/' ]]; then
					emsg="${lf}Option -l: invalid value '$OPTARG'. Must begin with /"
				fi
				log_dir="$OPTARG"
				if [[ ${log_dir:${#log_dir}} != '/' ]]; then
					log_dir="$log_dir/"
				fi
				;;
			r )
				case "$OPTARG" in
					'DVD_write' | 'DVD_verify' ) 
						restart="$OPTARG"
						;;
					* )
						emsg="${lf}Option -r: invalid argument '$OPTARG'"
						;;
				esac
				;;
			V )
				echo "$prgnam version $prg_ver"
				\exit 0
				;;
			w )
				writer_opt_flag='YES'
				DVD_dev="$OPTARG"
				;;
			* )
				emsg="${lf}Invalid option '$opt'"
		esac
	done

	# Test for extra arguments
	# ~~~~~~~~~~~~~~~~~~~~~~~~
	shift $(( $OPTIND-1 ))
	if [[ $* != '' ]]; then
        emsg="${lf}Invalid extra argument(s) '$*'"
	fi

	# Test for mandatory options not set
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	if [[ ${cfg_afn:-} = '' ]]; then
        emsg="${lf}Mandatory option -c not given"
	fi

	# Report any errors
	# ~~~~~~~~~~~~~~~~~
	if [[ $emsg != '' ]]; then
		echo "$emsg" >&2
        usage
        \exit 1
	fi
	
	# Linux executables
	# ~~~~~~~~~~~~~~~~~
	export PATH=/usr/bin 				# $PATH is not used by this script but growisofs needs it to find genisofs
	awk='/usr/bin/awk'
	chmod='/bin/chmod'
	cksum='/usr/bin/cksum'
	dar_static='/usr/bin/dar_static'
	dar='/usr/bin/dar'
	date='/bin/date'
	df='/bin/df'
	dvd_rw_format='/usr/bin/dvd+rw-format'
	dvd_rw_mediainfo='/usr/bin/dvd+rw-mediainfo'
	eject='/usr/bin/eject'
	grep='/bin/grep'
	growisofs='/usr/bin/growisofs'
	lvcreate='/sbin/lvcreate'
	lvremove='/sbin/lvremove'
	lvs='/sbin/lvs'
	mkdir='/bin/mkdir'
	mktemp='/bin/mktemp'
	mount='/bin/mount'
	par2='/usr/bin/par2'
	ps='/bin/ps'
	readlink='/bin/readlink'
	rm='/bin/rm'
	script='/usr/bin/script'
	sleep='/bin/sleep'
	stty='/bin/stty'
	tail='/usr/bin/tail'
	umount='/bin/umount'

	# Ensure they all exist except par2 which is optional
	# @@ maybe the LVM commands should be optional, too
	buf="$( ck_file $awk:fx $chmod:fx $cksum:fx $dar_static:fx $dar:fx $date:fx $df:fx $dvd_rw_format:fx $dvd_rw_mediainfo:fx $eject:fx $grep:fx $growisofs:fx $lvcreate:fx $lvremove:fx $lvs:fx $mkdir:fx $mktemp:fx $mount:fx $ps:fx $readlink:fx $rm:fx $script:fx $sleep:fx $stty:fx $tail:fx $umount:fx 2>&1 )"
	if [[ $? -ne 0 ]] ; then
		echo "$prgnam: terminating on Linux executable problem(s):$lf$buf" >&2
		\exit 1
	fi

	# Note whether being run from a terminal
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# The "standard" test is to check $PS1 but this test is more reliable
	buf="$( $ps -p $$ -o tty 2>&1 )"
	case $buf in
		*TT*-* )
			interactive_flag='NO'
			;;
		*TT* )
			interactive_flag='YES'
			;;
		* )
			echo "$prgnam: Unable to determine if being run interactively.  ps output was: $buf" >&2
			\exit 1
	esac

	# Set default file and directory creation permissions
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# Chosen for security
	umask 0077

	# Initialise log file
	# ~~~~~~~~~~~~~~~~~~~
	# First part of the log file name is the config file name with any symlinks resolved
	cfg_fn="$( $readlink $cfg_afn 2>&1 )"
	if [[ $cfg_fn = '' ]]; then
		cfg_fn="${cfg_afn##*/}"
	fi
	buf="$( ck_file $log_dir:dwx 2>&1 )"
	if [[ $? -ne 0 ]] ; then
		echo "$prgnam: Terminating on log directory problem:$lf$buf" >&2
		\exit 1
	fi
	timestamp="$( $date +%y-%m-%d@%H:%M:%S)"
	log_afn="${log_dir}${cfg_fn}_$timestamp"

	# Up to this point any messages have been given using echo followed by \exit 1.  Now 
	# the essentials for call_msg() and finalise() have been established, all future messages 
	# will be sent using call_msg() and error mesages will then call finalise().

	call_msg 'I' "$prgnam version $prg_ver started with command line '$args'.  Logging to '$log_afn'"

	fct "${FUNCNAME[ 0 ]}" 'started (this message delayed until logging initialised)'

	# Exit if not running interactively
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	if [[ $interactive_flag != 'YES' ]]; then
		call_msg 'E' 'Not running interactively'
	fi

	# Create temporary directories and files
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	tmp_dir="$( $mktemp -d "/tmp/$prgnam.XXX" 2>&1 )"
	rc=$?
	if [[ $rc -ne 0 ]]; then
		call_msg 'E' "Unable to create temporary directory:$lf$tmp_dir"
	fi
	tmp_dir_created_flag='YES'
	tmp_dir="$tmp_dir/"
	snapshot_mnt_base_dir="${tmp_dir}snapshots/"
	buf="$($mkdir "$snapshot_mnt_base_dir" 2>&1)"
	if [[ $? -ne 0 ]]; then
  	 	call_msg 'E' "Unable to establish snapshots base directory '$snapshot_mnt_base_dir'.  $mkdir output was$lf$buf" 
	fi
	DVD_mnt_dir="${tmp_dir}mnt/"
	buf="$($mkdir "$DVD_mnt_dir" 2>&1)"
	if [[ $? -ne 0 ]]; then
  	 	call_msg 'E' "Unable to establish DVD mount point directory '$DVD_mnt_dir'.  $mkdir output was$lf$buf" 
	fi
	tmp_afn="${tmp_dir}tmp"
	tmp_sh_afn="${tmp_dir}tmp_sh"
	echo '' > "$tmp_sh_afn"
	$chmod '700' "$tmp_sh_afn"

	# Set defaults that may be overriden by the configuration file
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	bk_dir='/var/opt/MyBackup.sh/'
	drct_dir_tree=''
	global_message=''

	# Parse configuration file
	# ~~~~~~~~~~~~~~~~~~~~~~~~
	parse_cfg "$cfg_afn"

	# Post-configuration error traps, initialisation and reporting
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	buf="$(ck_file "$bk_dir":drwx 2>&1)"
	if [[ $? -ne 0 ]]; then							# Backup directory not OK so silently try to create it
		buf="$($mkdir -p "$bk_dir" 2>&1)"
		if [[ $? -ne 0 ]]; then
  	 		call_msg 'E' "Unable to establish backup directory '$bk_dir'.  $mkdir -p output was$lf$buf" 
		fi
	fi

	buf="$( $readlink $DVD_dev 2>&1 )"		 		# Resolve any DVD device file symlinks
	msgpart=' (after resolving symlinks)'
	case "$buf" in
		"" )
			msgpart=''
			;;
		'/'* )
			DVD_dev="$buf"
			;;
		* )
			DVD_dev="${DVD_dev%/*}/$buf"
	esac
	buf="$(ck_file $DVD_dev:brw 2>&1)"
	if [[ "$buf" != '' ]];then
  	 	call_msg 'E' "DVD writer problem$msgpart: $buf" 
	fi
	call_msg 'I' "DVD writer$msgpart: $DVD_dev"

	if [[ $drct_dir_tree != '' ]]; then
		buf="$(ck_file "$drct_dir_tree":drx 2>&1)"
		if [[ $? -eq 0 ]]; then
			drct_dir_tree="${drct_dir_tree// /\\ }"		# escapes any spaces to facilitate later parsing unquoted
		else
  	 		call_msg 'W' "Direct directory tree '$drct_dir_tree': $buf"
			drct_dir_tree=''
		fi
	fi

	if [[ $global_message != '' ]]; then
		tty_msg "$global_message"
	fi

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of function initialise

#--------------------------
# Name: mount_DVD
# Purpose: mounts DVD at mount point in temporary directory structure
#--------------------------
function mount_DVD {

	fct "${FUNCNAME[ 0 ]}" 'started'

	local buf
	
	# Mount DVD
	# ~~~~~~~~~
	$umount $DVD_dev >/dev/null 2>&1 	# crude and precautionary
	buf="$( $mount -o ro -t iso9660 $DVD_dev $DVD_mnt_dir 2>&1 )"		# Do not rely on file system type auto-deetection
	if [[ $buf != '' ]]; then
		call_msg 'I' 'Last 300 lines of /var/log/kern.log:'
		call_msg -nt 'I' "$($tail -300 /var/log/kern.log)"
		call_msg 'E' "unable to mount DVD $DVD_dev at temporary mount point $DVD_mnt_dir: $buf"
	fi
	DVD_mounted_flag='YES'

	call_msg 'I' "DEBUG for https://bugs.launchpad.net/ubuntu/+source/linux/+bug/228624: After mounting DVD, /sys/block/sr0/size contains: $(/bin/cat /sys/block/sr0/size)" 

	fct "${FUNCNAME[ 0 ]}" 'returning'
	return 0

}  # end of function mount_DVD

#--------------------------
# Name: msg
# Purpose: generalised messaging interface
# Usage: msg [ -l logfile ] [ -n l|s|t ] class msg_text
#    -l logs any error messages to logfile unless -n l overrides
#    -n suppress:
#		l writing to log
#		s writing to stdout or stderr
#		t writing timestamp and I|W|E to log
#    class must be one of I, W or E indicating Information, Warning or Error
#    msg_text is the text of the message
# Return code:  always zero (exits on error)
#--------------------------
function msg {

    local args buf indent line logfile message_text no_log_flag no_screen_flag no_timestamp_flag preamble

    # Parse any options
    # ~~~~~~~~~~~~~~~~~
	args="${@:-}"
	OPTIND=0								# Don't inherit a value
	no_log_flag='NO'
	no_screen_flag='NO'
	no_timestamp_flag='NO'
	while getopts l:n: opt 2>/dev/null
	do
		case $opt in
			l)
				logfile="$OPTARG"
				;;
			n)
				case ${OPTARG:=} in
					'l' )
						no_log_flag='YES'
						;;
					's' )
						no_screen_flag='YES'
						;;
					't' )
						no_timestamp_flag='YES'
						;;
					* )
			    		echo "$prgnam: msg: invalid -n option argument, '$OPTARG'" >&2
			    		\exit 1
				esac
				;;
			* )
			    echo "$prgnam: msg: invalid option, '$opt'" >&2
			    \exit 1
			    ;;
		esac
	done
	shift $(( $OPTIND-1 ))

    # Parse the mandatory positional parameters (class and message)
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    class="${1:-}"
    case "$class" in 
    	I | 'E' ) 
	    	;;
    	'W' ) 
			global_warning_flag='YES'
	    	;;
	    * )
	    	echo "$prgnam: msg: invalid arguments: '$args'" >&2
	    	\exit 1
    esac

    message_text="$2"

    # Write to log if required
    # ~~~~~~~~~~~~~~~~~~~~~~~~
    if [[ $logfile != ''  && $no_log_flag == 'NO' ]]
    then
		if [[ $no_timestamp_flag == 'NO' ]]; then
			buf="$( $date +%H:%M:%S ) $class "
		else
			buf=''
		fi
		echo "$buf$message_text" >> $logfile
		if [[ $? -ne 0 ]]
		then
			echo "$prgnam: msg: unable to write to log file '$logfile'" >&2
			\exit 1
		fi
	fi
	
	# Write to stdout or stderr if not suppressed
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	if [[ $no_screen_flag = 'NO' ]]; then
	    case "$class" in 
	    	I  ) 
				echo "$message_text" >&1
		    	;;
		    W  ) 
				echo "WARNING: $message_text" >&1
		    	;;
		    E )
				echo "ERROR: $message_text" >&2
		    	;;
	    esac
	fi

    return 0

}  #  end of function msg

#--------------------------
# Name: parse_cfg
# Purpose: parses the configuration file
# $1 - pathname of file to parse
#--------------------------
function parse_cfg {

	fct "${FUNCNAME[ 0 ]}" 'started'

	local bk_dir_flag buf cfg_afn data dir_tree_flag drct_dir_tree_flag DVD_dev_flag excludes_flag global_par2_flag keyword par2_executable_flag processing_stanzas_flag prunes_flag
	local -a array
	
	cfg_afn="$1"

	# Does the file exist?
	# ~~~~~~~~~~~~~~~~~~~~
	buf="$( ck_file $cfg_afn:fr 2>&1 )"
	if [[ $buf != '' ]]
	then
		call_msg 'E' "Terminating on configuration file problem:$lf$buf"
		\exit 1
	fi

	# Initialisation
	# ~~~~~~~~~~~~~~
	emsg=''
	n_dar_jobs=0
	processing_stanzas_flag='NO'
	ck_file $par2:fx 2>&1
	if [[ $? -eq 0 ]] ; then
		global_par2_flag='YES'
		par2_executable_flag='YES'
	else
		global_par2_flag='NO'
		par2_executable_flag='NO'
	fi
	exec 3< $cfg_afn 								# set up the config file for reading on file descriptor 3

	# For each line in the config file ...
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	while read -u 3 buf								# for each line of the config file
	do
		buf="${buf%%#*}"							# strip any comment
		buf="${buf%%*( 	)}"							# strip any trailing spaces and tabs
		buf="${buf##*( 	)}"							# strip any leading spaces and tabs
		if [[ $buf = '' ]]; then
			continue								# empty line
		fi
		keyword="${buf%%:*}"
		keyword="${keyword%%*( 	)}"					# strip any trailing spaces and tabs
		data="${buf#*:}"							# strip to first : (keyword +)
		data="${data##*([ 	])}"					# strip any leading spaces and tabs
		case "$keyword" in
			'Backup directory' )
				if [[ ${bk_dir_flag:-} = 'YES' ]]; then
					emsg="$emsg${lf}  Backup directory specified again"
				fi
				bk_dir_flag='YES'
				bk_dir="${data%%*(/)}/"  			# silently ensure a single trailing slash
				;;
			'Basename' )
				if [[ $n_dar_jobs -gt 0 ]]; then
					if [[ "$dir_tree_flag$excludes_flag$prunes_flag" != 'YESYESYES' ]];then
						emsg="$emsg${lf}  Stanza ${basenames[ $n_dar_jobs ]} incomplete"
					fi
				fi
				: $(( n_dar_jobs++ ))
				dir_tree_flag='NO'
				excludes_flag='NO'
				processing_stanzas_flag='YES'
				prunes_flag='NO'
				basenames[ $n_dar_jobs ]="$data"
				message[ $n_dar_jobs ]=''
				par2_flag[ $n_dar_jobs ]="$global_par2_flag"
				;;
			'Direct directory tree' )
				if [[ ${drct_dir_tree_flag:-} = 'YES' ]]; then
					emsg="$emsg${lf}  Direct directory tree specified again"
				fi
				drct_dir_tree_flag='YES'
				eval array=( "$data" )					# parses any quoted words
				if [[ ${#array[*]} -ne 1 ]]; then
					emsg="$emsg${lf}  Direct directory tree data is not one word"
				fi
				drct_dir_tree="${array[ 0 ]}"
				;;
			'DVD writer device' )
				if [[ ${DVD_dev_flag:-} = 'YES' ]]; then
					emsg="$emsg${lf}  DVD writer device specified again"
				fi
				DVD_dev_flag='YES'
				if [[ $writer_opt_flag = 'NO' ]]; then
					DVD_dev="$data"
				else
					call_msg 'I' "DVD writer specified on command line with -w option; DVD writer in configuration file $cfg_afn ignored"
				fi
				;;
			'Directory tree (fs_root)' )
				if [[ $dir_tree_flag = 'YES' ]]; then
					emsg="$emsg${lf}  Stanza ${basenames[ $n_dar_jobs ]}: Directory tree (fs_root) specified again"
				fi
				dir_tree_flag='YES'
				eval array=( "$data" )					# parses any quoted words
				if [[ ${#array[*]} -ne 1 ]]; then
					emsg="$emsg${lf}  Directory tree data is not one word"
				fi
				cfg_fs_roots[ $n_dar_jobs ]="${array[0]}"
				;;
			'Excludes' )
				if [[ $excludes_flag = 'YES' ]]; then
					emsg="$emsg${lf}  Stanza ${basenames[ $n_dar_jobs ]}: Excludes specified again"
				fi
				excludes_flag='YES'
				excludes[ $n_dar_jobs ]="$data"
				;;
			'Message' )
				if [[ $processing_stanzas_flag = 'NO' ]]; then
					global_message="$data"
				else
					message[ $n_dar_jobs ]="$data"
				fi
				;;
			'par2' )
				case "$data" in
					'YES' | 'NO' )
						;;
					* )
						emsg="$emsg${lf}  par2 value not YES or NO"
				esac
				if [[ $data = 'YES' && $par2_executable_flag = 'NO' ]]; then
					emsg="$emsg${lf}  par2 requested but $par2 does not exist or is not executable"
				fi
				if [[ $processing_stanzas_flag = 'NO' ]]; then
					global_par2_flag="$data"
				else
					par2_flag[ $n_dar_jobs ]="$data"
				fi
				;;
			'Snapshot size (KiB)' )
				# Was unable to get case with patterns like +([0-9]) to work (got syntax error near unexpected token `(') despite shopt -s extglob
				buf="${data//,/}"
				buf="${buf//[0-9]/}"
				if [[ "$buf" != '' ]];then
					emsg="$emsg${lf}  Snapshot size $data is not an unsigned integer"
				fi
				snapshot_sizes[ $n_dar_jobs ]="$data"
				;;
			'Prunes' )
				if [[ $prunes_flag = 'YES' ]]; then
					emsg="$emsg${lf}  Stanza ${basenames[ $n_dar_jobs ]}: Prunes specified again"
				fi
				prunes_flag='YES'
				prunes[ $n_dar_jobs ]="$data"
				;;
			* )
				emsg="$emsg${lf}  Unrecognised keyword. '$keyword'"
				;;
		esac

	done
	exec 3<&- # free file descriptor 3

	# Error traps
	# ~~~~~~~~~~~
	if [[ $n_dar_jobs -eq 0 ]]; then
		emsg="$emsg${lf}  No backup job stanzas found"
	else
		if [[ "$dir_tree_flag$excludes_flag$prunes_flag" != 'YESYESYES' ]];then
			emsg="$emsg${lf}  Stanza ${basenames[ $n_dar_jobs ]} incomplete"
		fi
	fi
	if [[ $emsg != '' ]]
	then
		call_msg 'E' "In configuration file $cfg_afn:$emsg"
	fi

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of function parse_cfg

#--------------------------
# Name: press_enter_to_continue
# Purpose: prints "Press enter to continue..." message and waits for user to do so
#--------------------------
function press_enter_to_continue {

	echo "Press enter to continue..."
	read

}  # end of function press_enter_to_continue

#--------------------------
# Name: report_dar_retval
# Purpose: reports on return value from dar
# Usage: report_dar_retval retval
#	where 
#		retval is the return value from dar
#--------------------------
function report_dar_retval {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local retval 

	retval=$?

	case $retval in
		0 )
		    ;;
		1 )
		    call_msg 'E' "Programming error in $prgnam: dar syntax error"
		    ;;
        2 )
 		    call_msg 'E' 'dar: hardware problem or lack of memory'
 		    ;;
        3 | 7 | 8 | 9 | 10 )
  		    call_msg 'E' "dar: internal dar problem; consult dar man page for return code $retval"
 		    ;;
        5 )
    		call_msg 'W' 'dar: file to be backed up could not be opened or read'
    		if [[ ${prgnam_retval:-} -lt 5 ]]; then
    		    prgnam_retval=5
    		fi
 		    ;;
        6 )
            # Should not be possible!  Code written here in case dar command altered to include -E or -F
    		call_msg 'W' 'dar: called program error'
    		if [[ ${prgnam_retval:-} -lt 6 ]]; then
    		    prgnam_retval=6
    		fi
 		    ;;
        11 )
    		call_msg 'W' 'dar: detected at least one file changed while being backed up'
    		if [[ ${prgnam_retval:-} -lt 11 ]]; then
    		    prgnam_retval=11
    		fi
 		    ;;
		* )
			# In case further return values introduced to dar
  		    call_msg 'E' "dar: unexpected return code $retval.  $prgnam needs updating to suit"
 		    ;;
	esac

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of report_dar_retval

#--------------------------
# Name: set_up_snapshot
# Purpose: creates and mounts LVM snapshot of the file system containing the directory tree 
# 	to back up (the fs_root on dar's -R option)
# Usage: set_up_snapshot index
#	where 
#		index is the dar job index number
#--------------------------
function set_up_snapshot {

	fct "${FUNCNAME[ 0 ]}" "started with idx=$1"

    local buf cmd idx LVM_device LVM_snapshot_block_device size vg

	idx="$1"

	# Build alternative device name and snapshot device names
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	buf="${dev_afn#/dev/mapper/}"
	buf="${buf%% *}"
	vg="${buf%%-*}"
	vn="${buf#*-}"
	LVM_device="/dev/$vg/$vn"
	LVM_snapshot_device="${LVM_device}_snap"
	LVM_snapshot_block_device="/dev/mapper/${vg}-${vn}_snap"
	LVM_snapshot_devices[ $idx ]="$LVM_snapshot_device"

	# Build snapshot mount point name
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	snapshot_mnt_point="${snapshot_mnt_base_dir}$vg/$vn/"
	snapshot_mnt_points[ $idx ]="$snapshot_mnt_point"

	# Ensure snapshot volume exists
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	if [[ -b $LVM_snapshot_device ]]; then
		call_msg 'I' "LVM snapshot volume $LVM_snapshot_device already exists.  Continuing ..."
	else
		# Get size for snapshot volume
		# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		size="${snapshot_sizes[ $n_dar_jobs ]:-}"
		if [[ $size = '' ]]; then
			# Snapshot size was not specified in the configuration file so use the actual size of the LV for safety.
			# Units k (KiB) are chosen for precision and so they can be used on the lvcreate command
			buf="$($lvs --units k -o lv_size "$LVM_device" 2>&1)"
			if [[ $? -ne 0 ]]; then
			    call_msg 'E' "Unable to get logical volume information for '$LVM_device'.  $lvs output was$lf$buf" 
			fi
			buf="${buf##*LSize*([!0-9])}"
			size="${buf%%.*}"
		else
			size="${size//,/}"				# Silently remove any commas
		fi
	
		# Flush file system buffers
		# ~~~~~~~~~~~~~~~~~~~~~~~~~
		($sync; $sleep 1; $sync; $sleep 1) >/dev/null 2>&1
	
		# Create snapshot volume
		# ~~~~~~~~~~~~~~~~~~~~~~
		# This is made same size as original to be 100% sure it will be big enough.
		buf="$($lvcreate --size ${size}k --snapshot --name "$LVM_snapshot_device" "$LVM_device" 2>&1)"
		if [[ $? -ne 0 ]]; then
		    call_msg 'E' "Unable to create snapshot volume.  $lvcreate output was$lf$buf" 
		fi
		call_msg 'I' "LVM snapshot volume $LVM_snapshot_device created"
	fi

	#  Ensure snapshot mounted
	# ~~~~~~~~~~~~~~~~~~~~~~~~
	cmd="$df $LVM_snapshot_block_device"
	buf="$( $cmd 2>&1 )"
	case ${buf##* } in
		'/dev' )
			# Mount snapshot
			buf="$($mkdir -p "$snapshot_mnt_point" 2>&1)"
	 		if [[ $? -ne 0 ]]; then
	   		 	call_msg 'E' "Unable to establish snapshot mount point directory '$snapshot_mnt_point'.  $mkdir output was$lf$buf" 
			fi
			buf="$($mount -o ro "$LVM_snapshot_device" "$snapshot_mnt_point" 2>&1)"
			if [[ $? -ne 0 ]]; then
	   		 	call_msg 'E' "Unable to mount snapshot volume.  $mount output was$lf$buf" 
			fi
			snapshot_vol_mounted_flag[ $idx ]='YES'
			call_msg 'I' "LVM snapshot volume $LVM_snapshot_device mounted on $snapshot_mnt_point"
			;;
		"${snapshot_mnt_point%/}")
			call_msg 'I' "LVM snapshot volume $LVM_snapshot_device already mounted on $snapshot_mnt_point"
			;;
		* )
			call_msg 'E' "LVM snapshot volume $LVM_snapshot_device already mounted but not on $snapshot_mnt_point$lf$cmd output was$lf$buf"
			;;
	esac

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of set_up_snapshot

#--------------------------
# Name: tty_msg
# Purpose: puts message on /dev/tty and reads user's response.  If empty then continues.  If q or Q then quits
# Usage: tty_msg <message> <quit message>
# <quit message> is optional
# Return value:  always 0
#--------------------------
function tty_msg {

	fct "${FUNCNAME[ 0 ]}" 'started'

	local reply

	while true
	do
		echo "$1  (Press Enter to continue or Q to quit)" > /dev/tty
		read reply
		case "$reply" in
			'q' | 'Q' )
				call_msg 'I' "${2:-User chose to quit when asked '$1'}"
				finalise 0
				;;
			'' )
				break
		esac
	done	

	echo 'Continuing ...' > /dev/tty

	fct "${FUNCNAME[ 0 ]}" 'returning'
	return 0

}  # end of tty_msg

#--------------------------
# Name: umount_DVD
# Purpose: unmounts DVD
# Usage: not used by finalise() because sleep and DEBU message unwanted there and umount code is so simple
#--------------------------
function umount_DVD {

	# growisofs does not like the DVD to be mounted
	$sleep 5				# Intended to be long enough for any automounting system such as gnome-volume-manager to do its thing.
	"$umount" "$DVD_dev" 	# umount output is ignored because DVD may not be automounted and, if it is, the best action is to continue anyway
	call_msg 'I' "DEBUG for https://bugs.launchpad.net/ubuntu/+source/linux/+bug/228624: When DVD loaded (and unmounted), /sys/block/sr0/size contains: $(/bin/cat /sys/block/sr0/size)"

	return 0

}  # end of function umount_DVD

#--------------------------
# Name: usage
# Purpose: prints usage message
#--------------------------
function usage {
	fct "${FUNCNAME[ 0 ]}" 'started'

	echo "usage: $prgnam -c <cfg file> [-d] [-D format | no_format] [-h] [-l <log dir>] [-r DVD_write | DVD_verify] [-V] [-w <writer>]" >&2
	if [[ ${1:-} = '' ]]
	then
		echo "(use -h for help)" >&2
	else
		echo "  where:
    -c names the configuration file.  If it does not begin with / then it is prefixed with $cfg_dir
       See sample configuration file for more information.
    -d turns debugging trace on.
    -D controls DVD options: 
         no_format prevents DVD formatting
         format formats DVD
    	 Default: no formatting.
    -h prints this help and exits.
    -l names the log directory. Must begin with /. Default if not specified $log_dir
    -r restarts the tasks defined by the configuration file at:
         DVD write (DVD_write) 
           or 
         DVD verification (DVD_verify).
       Intended for use after either has failed.
    -V prints the program version and exits.
    -w names the DVD writer, overriding the writer specified in the configuration file. Example /dev/scd1.
" >&2
	fi

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of function usage

#--------------------------
# Name: verify_DVD
# Purpose: verifies dar files on DVD
#--------------------------
function verify_DVD {

	fct "${FUNCNAME[ 0 ]}" 'starting'

	local buf cksum_out err_flag i retval

	# Compare dar backup file(s) on DVD against on-disk versions
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	err_flag='NO'
	i=1
	while [[ $i -le "$n_dar_jobs" ]]
	do
		fn="${basenames[ $i ]}.1.dar"

		# Checksum on-DVD backup
		# ~~~~~~~~~~~~~~~~~~~~~~
		# Do this first because it is the most error prone
		buf="$( $cksum $DVD_mnt_dir$fn 2>&1 )"
		retval=$?
		if [[ $retval -ne 0 ]]; then
			# Gather evidence for DVD read failure
			case $buf in
				*'Input/output error'* )
					call_msg 'I' 'Last 300 lines of /var/log/kern.log:'
					call_msg -nt 'I' "$($tail -300 /var/log/kern.log)"
			esac
			call_msg 'E' "unable to checksum $DVD_mnt_dir$fn.  Output was:$lf$buf" 
		else
			cksum_out="${buf%% /*}" # strip file name
			call_msg 'I' "checksum of ${DVD_mnt_dir}$fn is: $cksum_out"
		fi
		
		# Checksum on-disk backup
		# ~~~~~~~~~~~~~~~~~~~~~~~
		buf="$( $cksum $bk_dir$fn 2>&1 )"
		retval=$?
		if [[ $retval -ne 0 ]]; then
			call_msg 'E' "unable to checksum $bk_dir$fn.  Output was:$lf$buf" 
		else
			buf="${buf%% /*}" # strip file name
			call_msg 'I' "checksum of ${bk_dir}$fn is: $buf"
		fi

		# Compare check sums
		# ~~~~~~~~~~~~~~~~~~
		if [[ $buf = $cksum_out ]]; then
			call_msg 'I' "dar file on DVD is the same as dar file on disk"
		else
			call_msg 'W' "dar file on DVD not the same as dar file on disk"
			err_flag='YES'
		fi

		: $(( i++ ))
	done

	if [[ $err_flag != 'NO' ]]; then
		call_msg 'E' "one or more dar files on DVD not same as on disk (see above)"
	fi

	fct "${FUNCNAME[ 0 ]}" 'returning'
	return 0

}  # end of function verify_DVD

#--------------------------
# Name: write_DVD
# Purpose: copies on-disk backups to DVD
# Usage: write_DVD 
#--------------------------
function write_DVD {

	fct "${FUNCNAME[ 0 ]}" 'started'

    local buf error_flag file_list idx log_msg

	# Build list of files to copy to DVD
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	file_list="$dar_static"					# So can restore on a system without dar installed
	ck_file /etc/lvm:drx >/dev/null 2>&1
	if [[ $? -eq 0 ]]; then
		file_list="$file_list LVM=/etc/lvm"
	fi
	if [[ ${drct_dir_tree:-} != '' ]]; then
		buf="${drct_dir_tree##*/}"
		file_list="$file_list $buf=$drct_dir_tree"
	fi
	idx=1
	while [[ $idx -le "$n_dar_jobs" ]]
	do
		file_list="$file_list ${bk_dir}${basenames[ $idx ]}.1.dar "
		if [[ ${par2_flag[ $idx ]} = 'YES' ]]; then
			file_list="$file_list $(echo "$bk_dir${basenames[ $idx ]}.1.dar"*'.par2')"
		fi
		: $(( idx++ ))
	done

	# Write shellscript to copy files to DVD
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	buf="#! /bin/bash$lf $growisofs -Z $DVD_dev -speed=1 -iso-level 4 -r -graft-points $file_list$lf# End of script"
	echo "$buf" > $tmp_sh_afn 	
	call_msg 'I' "Running on-the-fly script to copy files to DVD.  Script is:$lf$buf"

	# Run the shellscript, capturing its screen output in a temporary file
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	"$script" '-c' "$tmp_sh_afn" '-q' '-f' "$tmp_afn"

	# Analyse captured screen output
	# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	error_flag='NO'
	log_msg=''
	exec 3< $tmp_afn 												# Set up the temp file for reading on file descriptor 3
	while read -u 3 buf												# For each line of the script output to screen
	do
		buf="${buf%
}"
		case "$buf" in
			':-('* \
			| 'File'*'is larger than'* \
			| '-allow-limited-size was not specified'* )			# Error
				error_flag='YES'
				log_msg="$log_msg$lf$buf"
				;;
			'Executing'* \
			| 'About to execute'* \
			| 'Warning: Creating ISO-9660:'* \
			| 'Warning: ISO-9660 filenames'* \
			| *'already carries isofs!'* \
			| *'"Quick Grow" session...' \
			| *'Current Write Speed'* \
			| 'Total'* \
			| 'Path table'* \
			| 'Max brk space'* \
			| *'extents written'* \
			| 'builtin_dd:'* \
			| *'flushing cache'* \
			| *'stopping de-icing'* \
			| *'writing lead-out'* \
			) 														# Normal output
				log_msg="$log_msg$lf$buf"
				;;
			*'done, estimate finish'* )								# Normal output, not required in log
				;;
			* )														# Unexpected output
				error_flag='YES'
				log_msg="$log_msg${lf}UNEXPECTED: $buf"
				;;
		esac

	done
	exec 3<&- 														# Free file descriptor 3

	if [[ $error_flag = 'NO' ]]; then 
		call_msg -n s 'I' "Files copied to DVD OK:$log_msg"
	else
		call_msg 'W' "Unexpected output line(s) when copying files to DVD (prefixed with UNEXPECTED:):$log_msg"
	fi

	call_msg 'I' "DEBUG for https://bugs.launchpad.net/ubuntu/+source/linux/+bug/228624: After copying files to DVD, /sys/block/sr0/size contains: $(/bin/cat /sys/block/sr0/size)" 

	fct "${FUNCNAME[ 0 ]}" 'returning'

}  # end of write_DVD

#--------------------------
# Name: main
# Purpose: where it all happens
#--------------------------

# Configure script environment
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
set -o posix
set -o nounset
shopt -s extglob # allow extended pattern matching operators 

# Configure traps
# ~~~~~~~~~~~~~~~
# Set up essentials for finalise() and what it may call before setting traps (because finalise() may be called at any time after then)
debug='NO'
n_dar_jobs=0
prgnam=${0##*/}			# program name w/o path
global_warning_flag='NO'
trap 'finalise 129' 'HUP'
trap 'finalise 130' 'INT'
trap 'finalise 131' 'QUIT'
trap 'finalise 143' 'TERM'

# Initialise
# ~~~~~~~~~~
initialise "${@:-}"

# Back up each directory tree
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~
if [[ $restart = '' ]]; then
	back_up_dirtrees
fi

# DVD operations
# ~~~~~~~~~~~~~~
ensure_DVD_loaded
if [[ $DVD_format_flag = 'YES' ]]; then
	format_DVD
fi
umount_DVD
if [[ $restart != 'DVD_verify' ]]; then
	write_DVD
fi
mount_DVD
verify_DVD

# Finalise
# ~~~~~~~~
finalise 0

